import { NextRequest, NextResponse } from "next/server"
import { query, initDatabase } from "@/lib/db"
import { generateId, rowToUser } from "@/lib/utils/db-helpers"

export async function GET() {
  // Ensure database is initialized
  await initDatabase()
  try {
    const result = await query(
      `SELECT * FROM users ORDER BY "createdAt" DESC`
    )

    const users = result.rows.map(rowToUser)
    return NextResponse.json(users)
  } catch (error) {
    console.error("Error fetching users:", error)
    return NextResponse.json({ error: "Failed to fetch users" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  // Ensure database is initialized
  await initDatabase()
  
  try {
    const body = await request.json()
    const { email, name, image } = body

    if (!email || !name) {
      return NextResponse.json({ error: "Email and name are required" }, { status: 400 })
    }

    // Check if user already exists
    const existingResult = await query(
      `SELECT * FROM users WHERE email = $1`,
      [email]
    )

    if (existingResult.rows.length > 0) {
      return NextResponse.json(rowToUser(existingResult.rows[0]))
    }

    const id = generateId()
    const now = new Date()

    const result = await query(
      `INSERT INTO users (id, email, name, image, "createdAt", "updatedAt")
       VALUES ($1, $2, $3, $4, $5, $6)
       RETURNING *`,
      [id, email, name, image || null, now, now]
    )

    const user = rowToUser(result.rows[0])
    return NextResponse.json(user, { status: 201 })
  } catch (error) {
    console.error("Error creating user:", error)
    return NextResponse.json({ error: "Failed to create user" }, { status: 500 })
  }
}
