import { Pool } from 'pg'

let pool: Pool | undefined

// Default database configuration for development
const DEFAULT_DATABASE_URL = 'postgresql://postgres:postgres@localhost:5432/corex_dashboard?schema=public'

// Lazy initialization of the database pool
function getPool(): Pool {
  if (!pool) {
    const connectionString = process.env.DATABASE_URL || DEFAULT_DATABASE_URL

    pool = new Pool({
      connectionString,
      ssl: process.env.NODE_ENV === 'production' && process.env.DATABASE_URL?.includes('sslmode=require') 
        ? { rejectUnauthorized: false } 
        : false,
    })

    // Handle pool errors
    pool.on('error', (err) => {
      console.error('Unexpected error on idle client', err)
      // Don't exit in development, just log
      if (process.env.NODE_ENV === 'production') {
        process.exit(-1)
      }
    })
  }

  return pool
}

export { getPool as db }

// Helper function to query the database
export async function query(text: string, params?: unknown[]) {
  const dbPool = getPool()
  const start = Date.now()
  const res = await dbPool.query(text, params)
  const duration = Date.now() - start
  if (process.env.NODE_ENV === 'development') {
    console.log('Executed query', { text, duration, rows: res.rowCount })
  }
  return res
}

// Initialize database tables if they don't exist
export async function initDatabase() {
  try {
    const dbPool = getPool()
    
    // Check if tables exist
    const result = await dbPool.query(`
      SELECT EXISTS (
        SELECT FROM information_schema.tables 
        WHERE table_schema = 'public' 
        AND table_name = 'customers'
      );
    `)

    if (!result.rows[0].exists) {
      console.log('Initializing database tables...')
      
      // Read and execute init SQL
      const fs = await import('fs/promises')
      const path = await import('path')
      
      try {
        const sqlPath = path.join(process.cwd(), 'scripts', 'init-db.sql')
        const sql = await fs.readFile(sqlPath, 'utf-8')
        
        // Split by semicolon and execute each statement
        const statements = sql
          .split(';')
          .map(s => s.trim())
          .filter(s => s.length > 0 && !s.startsWith('--'))
        
        for (const statement of statements) {
          if (statement.trim()) {
            await dbPool.query(statement)
          }
        }
        
        console.log('Database tables initialized successfully')
      } catch (fileError) {
        console.error('Error reading init-db.sql, trying direct SQL...', fileError)
        // Fallback: execute SQL directly
        await dbPool.query(`
          CREATE TABLE IF NOT EXISTS users (
            id TEXT PRIMARY KEY,
            email TEXT NOT NULL UNIQUE,
            name TEXT NOT NULL,
            image TEXT,
            "createdAt" TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            "updatedAt" TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
          );

          CREATE TABLE IF NOT EXISTS customers (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            email TEXT,
            phone TEXT,
            company TEXT,
            address TEXT,
            notes TEXT,
            "createdAt" TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            "updatedAt" TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
          );

          CREATE TABLE IF NOT EXISTS time_entries (
            id TEXT PRIMARY KEY,
            description TEXT NOT NULL,
            "startTime" TIMESTAMP NOT NULL,
            "endTime" TIMESTAMP NOT NULL,
            duration INTEGER NOT NULL,
            "createdAt" TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            "updatedAt" TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            "customerId" TEXT NOT NULL REFERENCES customers(id) ON DELETE CASCADE ON UPDATE CASCADE,
            "userId" TEXT NOT NULL REFERENCES users(id) ON DELETE CASCADE ON UPDATE CASCADE
          );

          CREATE INDEX IF NOT EXISTS idx_time_entries_customer_id ON time_entries("customerId");
          CREATE INDEX IF NOT EXISTS idx_time_entries_user_id ON time_entries("userId");
          CREATE INDEX IF NOT EXISTS idx_time_entries_created_at ON time_entries("createdAt");
        `)

        // Create trigger function and triggers
        await dbPool.query(`
          CREATE OR REPLACE FUNCTION update_updated_at_column()
          RETURNS TRIGGER AS $$
          BEGIN
              NEW."updatedAt" = CURRENT_TIMESTAMP;
              RETURN NEW;
          END;
          $$ language 'plpgsql';
        `)

        await dbPool.query(`
          DROP TRIGGER IF EXISTS update_users_updated_at ON users;
          CREATE TRIGGER update_users_updated_at BEFORE UPDATE ON users
              FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();
        `)

        await dbPool.query(`
          DROP TRIGGER IF EXISTS update_customers_updated_at ON customers;
          CREATE TRIGGER update_customers_updated_at BEFORE UPDATE ON customers
              FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();
        `)

        await dbPool.query(`
          DROP TRIGGER IF EXISTS update_time_entries_updated_at ON time_entries;
          CREATE TRIGGER update_time_entries_updated_at BEFORE UPDATE ON time_entries
              FOR EACH ROW EXECUTE FUNCTION update_updated_at_column();
        `)

        console.log('Database tables initialized successfully (fallback method)')
      }
    }
  } catch (error) {
    console.error('Error initializing database:', error)
    // Don't throw - allow app to continue, database might not be available yet
  }
}
