import { NextRequest, NextResponse } from "next/server"
import { query, initDatabase } from "@/lib/db"
import { generateId, rowToCustomer } from "@/lib/utils/db-helpers"

export async function GET(request: NextRequest) {
  // Ensure database is initialized
  await initDatabase()
  try {
    const searchParams = request.nextUrl.searchParams
    const search = searchParams.get("search")

    let customersQuery = `
      SELECT c.*
      FROM customers c
    `

    const params: string[] = []

    if (search) {
      customersQuery += ` WHERE c.name ILIKE $1 OR c.email ILIKE $1 OR c.company ILIKE $1`
      params.push(`%${search}%`)
    }

    customersQuery += ` ORDER BY c."createdAt" DESC`

    const result = await query(customersQuery, params.length > 0 ? params : undefined)
    
    // Get time entries for each customer
    const customers = await Promise.all(
      result.rows.map(async (row) => {
        const customer = rowToCustomer(row)
        const timeEntriesResult = await query(
          `SELECT te.*, 
            json_build_object('name', u.name, 'email', u.email) as user
           FROM time_entries te
           LEFT JOIN users u ON u.id = te."userId"
           WHERE te."customerId" = $1
           ORDER BY te."createdAt" DESC`,
          [customer.id]
        )
        const timeEntries = timeEntriesResult.rows.map((entry: {
          id: string
          description: string
          startTime: Date
          endTime: Date
          duration: number
          createdAt: Date
          updatedAt: Date
          customerId: string
          userId: string
          user: unknown
        }) => ({
          id: entry.id,
          description: entry.description,
          startTime: entry.startTime,
          endTime: entry.endTime,
          duration: entry.duration,
          createdAt: entry.createdAt,
          updatedAt: entry.updatedAt,
          customerId: entry.customerId,
          userId: entry.userId,
          user: typeof entry.user === 'string' ? JSON.parse(entry.user) : entry.user,
        }))
        return {
          ...customer,
          timeEntries,
        }
      })
    )

    return NextResponse.json(customers)
  } catch (error) {
    console.error("Error fetching customers:", error)
    return NextResponse.json({ error: "Failed to fetch customers" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  // Ensure database is initialized
  await initDatabase()
  
  try {
    const body = await request.json()
    const { name, email, phone, company, address, notes } = body

    if (!name || (typeof name === 'string' && name.trim() === "")) {
      return NextResponse.json({ error: "Name is required" }, { status: 400 })
    }

    try {
      const id = generateId()
      const now = new Date()

      const result = await query(
        `INSERT INTO customers (id, name, email, phone, company, address, notes, "createdAt", "updatedAt")
         VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
         RETURNING *`,
        [
          id,
          typeof name === 'string' ? name.trim() : name,
          email && typeof email === 'string' && email.trim() !== "" ? email.trim() : null,
          phone && typeof phone === 'string' && phone.trim() !== "" ? phone.trim() : null,
          company && typeof company === 'string' && company.trim() !== "" ? company.trim() : null,
          address && typeof address === 'string' && address.trim() !== "" ? address.trim() : null,
          notes && typeof notes === 'string' && notes.trim() !== "" ? notes.trim() : null,
          now,
          now,
        ]
      )

      const customer = rowToCustomer(result.rows[0])
      return NextResponse.json(customer, { status: 201 })
    } catch (dbError: unknown) {
      console.error("Database error creating customer:", dbError)
      const errorMessage = dbError instanceof Error ? dbError.message : "Database error occurred"
      const errorCode = (dbError as { code?: string })?.code
      return NextResponse.json(
        { 
          error: errorMessage,
          code: errorCode,
          details: process.env.NODE_ENV === 'development' && dbError instanceof Error ? dbError.stack : undefined
        }, 
        { status: 500 }
      )
    }
  } catch (error: unknown) {
    console.error("Error in POST /api/customers:", error)
    const errorMessage = error instanceof Error ? error.message : String(error)
    return NextResponse.json(
      { 
        error: errorMessage,
        details: process.env.NODE_ENV === 'development' && error instanceof Error ? error.stack : undefined
      }, 
      { status: 500 }
    )
  }
}
