"use client"

import { useState, useEffect } from "react"
import { Logo } from "@/components/Logo"
import { ThemeToggle } from "@/components/theme-toggle"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select } from "@/components/ui/select"
import { Play, Pause, Square, Clock, Filter, ArrowUpDown, Calendar, User } from "lucide-react"
import Link from "next/link"

interface Customer {
  id: string
  name: string
  company?: string
}

interface TimeEntry {
  id: string
  description: string
  startTime: string
  endTime: string
  duration: number
  createdAt: string
  customer: {
    id: string
    name: string
    company?: string
  }
  user: {
    name: string
    email: string
  }
}

type SortField = "date" | "duration" | "description"
type SortDirection = "asc" | "desc"

// TODO: Replace with actual user from authentication
const CURRENT_USER_ID = "temp-user-id"

export default function Zeiterfassung() {
  const [isRunning, setIsRunning] = useState(false)
  const [elapsedTime, setElapsedTime] = useState(0)
  const [pausedTime, setPausedTime] = useState(0)
  const [description, setDescription] = useState("")
  const [customerId, setCustomerId] = useState("")
  const [startTime, setStartTime] = useState<Date | null>(null)
  const [entries, setEntries] = useState<TimeEntry[]>([])
  const [customers, setCustomers] = useState<Customer[]>([])
  const [loading, setLoading] = useState(true)
  const [filterDate, setFilterDate] = useState("")
  const [filterCustomer, setFilterCustomer] = useState("")
  const [sortField, setSortField] = useState<SortField>("date")
  const [sortDirection, setSortDirection] = useState<SortDirection>("desc")
  const [showManualEntry, setShowManualEntry] = useState(false)
  const [manualHours, setManualHours] = useState("")

  useEffect(() => {
    fetchCustomers()
    fetchTimeEntries()
    // Create temp user if not exists
    ensureCurrentUser()
  }, [])

  const ensureCurrentUser = async () => {
    try {
      await fetch("/api/users", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          email: "temp@corexmanagement.de",
          name: "Temporärer Benutzer",
        }),
      })
    } catch (error) {
      console.error("Error ensuring user:", error)
    }
  }

  const fetchCustomers = async () => {
    try {
      const response = await fetch("/api/customers")
      if (response.ok) {
        const data = await response.json()
        setCustomers(data)
      }
    } catch (error) {
      console.error("Error fetching customers:", error)
    }
  }

  const fetchTimeEntries = async () => {
    try {
      setLoading(true)
      const response = await fetch("/api/time-entries")
      if (response.ok) {
        const data = await response.json()
        setEntries(data)
      }
    } catch (error) {
      console.error("Error fetching time entries:", error)
    } finally {
      setLoading(false)
    }
  }

  // Timer logic
  useEffect(() => {
    let interval: NodeJS.Timeout | null = null
    if (isRunning && startTime) {
      interval = setInterval(() => {
        const now = new Date()
        const diff = Math.floor((now.getTime() - startTime.getTime()) / 1000)
        setElapsedTime(pausedTime + diff)
      }, 1000)
    } else if (!isRunning && pausedTime > 0) {
      setElapsedTime(pausedTime)
    }
    return () => {
      if (interval) clearInterval(interval)
    }
  }, [isRunning, startTime, pausedTime])

  const formatTime = (seconds: number) => {
    const hours = Math.floor(seconds / 3600)
    const minutes = Math.floor((seconds % 3600) / 60)
    const secs = seconds % 60
    return `${hours.toString().padStart(2, "0")}:${minutes.toString().padStart(2, "0")}:${secs.toString().padStart(2, "0")}`
  }

  const startTimer = () => {
    if (!customerId) {
      alert("Bitte wählen Sie zuerst einen Kunden aus")
      return
    }
    const now = new Date()
    setStartTime(now)
    setIsRunning(true)
  }

  const pauseTimer = () => {
    if (startTime && isRunning) {
      const now = new Date()
      const currentDiff = Math.floor((now.getTime() - startTime.getTime()) / 1000)
      setPausedTime(prev => prev + currentDiff)
      setStartTime(null)
    }
    setIsRunning(false)
  }

  const resumeTimer = () => {
    if (!isRunning) {
      const now = new Date()
      setStartTime(now)
      setIsRunning(true)
    }
  }

  const stopTimer = async () => {
    if ((startTime || pausedTime > 0) && customerId) {
      const endTime = new Date()
      let duration = pausedTime
      
      if (startTime) {
        const currentDiff = Math.floor((endTime.getTime() - startTime.getTime()) / 1000)
        duration += currentDiff
      }
      
      const actualStartTime = startTime || new Date(endTime.getTime() - duration * 1000)
      
      try {
        // Get user ID first
        const usersResponse = await fetch("/api/users")
        const users = await usersResponse.json() as Array<{ id: string; email: string }>
        const userId = users.find((u) => u.email === "temp@corexmanagement.de")?.id || CURRENT_USER_ID

        const response = await fetch("/api/time-entries", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            description: description || "Keine Beschreibung",
            startTime: actualStartTime.toISOString(),
            endTime: endTime.toISOString(),
            duration,
            customerId,
            userId,
          }),
        })

        if (response.ok) {
          await fetchTimeEntries()
          setIsRunning(false)
          setStartTime(null)
          setElapsedTime(0)
          setPausedTime(0)
          setDescription("")
          setCustomerId("")
        }
      } catch (error) {
        console.error("Error saving time entry:", error)
        alert("Fehler beim Speichern des Zeiteintrags")
      }
    }
  }

  const addManualEntry = async () => {
    if (!description || !customerId || !manualHours) return

    const hours = parseFloat(manualHours.replace(",", "."))
    if (isNaN(hours) || hours <= 0) {
      alert("Bitte geben Sie eine gültige Stundenzahl ein (z.B. 1.5 für 1 Stunde 30 Minuten)")
      return
    }

    const now = new Date()
    const duration = Math.floor(hours * 3600)
    const startTime = new Date(now.getTime() - duration * 1000)

    try {
      const usersResponse = await fetch("/api/users")
      const users = await usersResponse.json() as Array<{ id: string; email: string }>
      const userId = users.find((u) => u.email === "temp@corexmanagement.de")?.id || CURRENT_USER_ID

      const response = await fetch("/api/time-entries", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          description,
          startTime: startTime.toISOString(),
          endTime: now.toISOString(),
          duration,
          customerId,
          userId,
        }),
      })

      if (response.ok) {
        await fetchTimeEntries()
        setDescription("")
        setCustomerId("")
        setManualHours("")
        setShowManualEntry(false)
      }
    } catch (error) {
      console.error("Error saving manual entry:", error)
      alert("Fehler beim Speichern des Zeiteintrags")
    }
  }

  const deleteEntry = async (id: string) => {
    if (!confirm("Eintrag wirklich löschen?")) return

    try {
      const response = await fetch(`/api/time-entries/${id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        await fetchTimeEntries()
      }
    } catch (error) {
      console.error("Error deleting entry:", error)
      alert("Fehler beim Löschen des Eintrags")
    }
  }

  // Filter and sort entries
  const filteredAndSortedEntries = entries
    .filter((entry) => {
      if (filterDate) {
        const entryDate = new Date(entry.startTime).toISOString().split("T")[0]
        if (entryDate !== filterDate) return false
      }
      if (filterCustomer && entry.customer.id !== filterCustomer) {
        return false
      }
      return true
    })
    .sort((a, b) => {
      let comparison = 0
      switch (sortField) {
        case "date":
          comparison = new Date(a.startTime).getTime() - new Date(b.startTime).getTime()
          break
        case "duration":
          comparison = a.duration - b.duration
          break
        case "description":
          comparison = a.description.localeCompare(b.description)
          break
      }
      return sortDirection === "asc" ? comparison : -comparison
    })

  const toggleSort = (field: SortField) => {
    if (sortField === field) {
      setSortDirection(sortDirection === "asc" ? "desc" : "asc")
    } else {
      setSortField(field)
      setSortDirection("desc")
    }
  }

  const totalTime = filteredAndSortedEntries.reduce((sum, entry) => sum + entry.duration, 0)

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b bg-card">
        <div className="container mx-auto flex h-16 items-center justify-between px-4">
          <Logo className="h-8" />
          <div className="flex items-center gap-4">
            <span className="text-sm text-muted-foreground">Mitarbeiter Dashboard</span>
            <ThemeToggle />
          </div>
        </div>
      </header>

      <main className="container mx-auto px-4 py-8 max-w-6xl">
        <div className="mb-8">
          <div className="flex items-center gap-4 mb-2">
            <Link href="/">
              <Button variant="ghost" size="sm">← Zurück</Button>
            </Link>
          </div>
          <h1 className="text-3xl font-bold mb-2">Zeiterfassung</h1>
          <p className="text-muted-foreground">
            Erfassen und verwalten Sie Ihre Arbeitszeit
          </p>
        </div>

        {/* Timer Card */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Clock className="h-5 w-5" />
              Timer
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="text-center">
              <div className="text-5xl font-mono font-bold mb-4">
                {formatTime(isRunning && startTime ? elapsedTime : pausedTime)}
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <Label htmlFor="customer">Kunde *</Label>
                <Select
                  id="customer"
                  value={customerId}
                  onChange={(e) => setCustomerId(e.target.value)}
                  disabled={isRunning}
                  required
                >
                  <option value="">Kunde auswählen</option>
                  {customers.map((customer) => (
                    <option key={customer.id} value={customer.id}>
                      {customer.name} {customer.company && `(${customer.company})`}
                    </option>
                  ))}
                </Select>
              </div>
              <div>
                <Label htmlFor="description">Beschreibung *</Label>
                <Input
                  id="description"
                  placeholder="Was arbeiten Sie an?"
                  value={description}
                  onChange={(e) => setDescription(e.target.value)}
                  disabled={isRunning}
                  required
                />
              </div>
            </div>

            <div className="flex gap-2 flex-wrap">
              {!isRunning && !startTime && pausedTime === 0 ? (
                <Button onClick={startTimer} size="lg" className="flex-1 min-w-[120px]">
                  <Play className="h-4 w-4 mr-2" />
                  Start
                </Button>
              ) : !isRunning ? (
                <>
                  <Button onClick={resumeTimer} size="lg" className="flex-1 min-w-[120px]">
                    <Play className="h-4 w-4 mr-2" />
                    Weiter
                  </Button>
                  <Button onClick={stopTimer} variant="destructive" size="lg" className="flex-1 min-w-[120px]">
                    <Square className="h-4 w-4 mr-2" />
                    Stoppen
                  </Button>
                </>
              ) : (
                <>
                  <Button onClick={pauseTimer} variant="outline" size="lg" className="flex-1 min-w-[120px]">
                    <Pause className="h-4 w-4 mr-2" />
                    Pause
                  </Button>
                  <Button onClick={stopTimer} variant="destructive" size="lg" className="flex-1 min-w-[120px]">
                    <Square className="h-4 w-4 mr-2" />
                    Stoppen
                  </Button>
                </>
              )}
              <Button 
                onClick={() => setShowManualEntry(!showManualEntry)} 
                variant="secondary" 
                size="lg" 
                className="flex-1 min-w-[120px]"
              >
                Manuell hinzufügen
              </Button>
            </div>

            {showManualEntry && (
              <div className="p-4 border rounded-lg bg-muted/50 space-y-4">
                <div>
                  <Label htmlFor="manualHours">Stunden (z.B. 1.5 für 1 Stunde 30 Minuten)</Label>
                  <Input
                    id="manualHours"
                    type="number"
                    step="0.25"
                    min="0.25"
                    placeholder="1.5"
                    value={manualHours}
                    onChange={(e) => setManualHours(e.target.value)}
                  />
                </div>
                <div className="flex gap-2">
                  <Button onClick={addManualEntry} className="flex-1" disabled={!customerId || !description}>
                    Eintrag hinzufügen
                  </Button>
                  <Button 
                    onClick={() => {
                      setShowManualEntry(false)
                      setManualHours("")
                    }} 
                    variant="outline"
                  >
                    Abbrechen
                  </Button>
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Filter and Sort */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filter & Sortierung
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-3">
              <div>
                <Label htmlFor="filterDate">Datum</Label>
                <Input
                  id="filterDate"
                  type="date"
                  value={filterDate}
                  onChange={(e) => setFilterDate(e.target.value)}
                />
              </div>
              <div>
                <Label htmlFor="filterCustomer">Kunde</Label>
                <Select
                  id="filterCustomer"
                  value={filterCustomer}
                  onChange={(e) => setFilterCustomer(e.target.value)}
                >
                  <option value="">Alle Kunden</option>
                  {customers.map((customer) => (
                    <option key={customer.id} value={customer.id}>
                      {customer.name}
                    </option>
                  ))}
                </Select>
              </div>
              <div>
                <Label>Sortierung</Label>
                <div className="flex gap-2">
                  <Button
                    variant={sortField === "date" ? "default" : "outline"}
                    size="sm"
                    onClick={() => toggleSort("date")}
                    className="flex-1"
                  >
                    <Calendar className="h-4 w-4 mr-1" />
                    Datum
                  </Button>
                  <Button
                    variant={sortField === "duration" ? "default" : "outline"}
                    size="sm"
                    onClick={() => toggleSort("duration")}
                    className="flex-1"
                  >
                    <Clock className="h-4 w-4 mr-1" />
                    Dauer
                  </Button>
                  <Button
                    variant={sortField === "description" ? "default" : "outline"}
                    size="sm"
                    onClick={() => toggleSort("description")}
                    className="flex-1"
                  >
                    <ArrowUpDown className="h-4 w-4 mr-1" />
                    Text
                  </Button>
                </div>
              </div>
            </div>
            {(filterDate || filterCustomer) && (
              <div className="mt-4">
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => {
                    setFilterDate("")
                    setFilterCustomer("")
                  }}
                >
                  Filter zurücksetzen
                </Button>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Entries List */}
        <Card>
          <CardHeader>
            <div className="flex justify-between items-center">
              <div>
                <CardTitle>Zeiteinträge</CardTitle>
                <CardDescription>
                  {filteredAndSortedEntries.length} Einträge
                  {totalTime > 0 && ` • Gesamt: ${formatTime(totalTime)}`}
                </CardDescription>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            {loading ? (
              <p className="text-center text-muted-foreground py-8">Lädt...</p>
            ) : filteredAndSortedEntries.length === 0 ? (
              <p className="text-center text-muted-foreground py-8">
                Noch keine Zeiteinträge vorhanden
              </p>
            ) : (
              <div className="space-y-2">
                {filteredAndSortedEntries.map((entry) => (
                  <div
                    key={entry.id}
                    className="flex items-center justify-between p-4 border rounded-lg hover:bg-accent/50 transition-colors"
                  >
                    <div className="flex-1">
                      <div className="font-medium">{entry.description}</div>
                      <div className="text-sm text-muted-foreground flex gap-4 mt-1">
                        <span>{new Date(entry.startTime).toLocaleString("de-DE")}</span>
                        <span>Kunde: {entry.customer.name}</span>
                        <span className="flex items-center gap-1">
                          <User className="h-3 w-3" />
                          {entry.user.name}
                        </span>
                      </div>
                    </div>
                    <div className="flex items-center gap-4">
                      <div className="text-lg font-mono font-semibold">
                        {formatTime(entry.duration)}
                      </div>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => deleteEntry(entry.id)}
                        className="text-destructive hover:text-destructive"
                      >
                        Löschen
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </main>
    </div>
  )
}
