import { NextRequest, NextResponse } from "next/server"
import { prisma } from "@/lib/prisma"

export async function GET(request: NextRequest) {
  try {
    const searchParams = request.nextUrl.searchParams
    const search = searchParams.get("search")

    const customers = await prisma.customer.findMany({
      where: search
        ? {
            OR: [
              { name: { contains: search } },
              { email: { contains: search } },
              { company: { contains: search } },
            ],
          }
        : undefined,
      orderBy: { createdAt: "desc" },
      include: {
        timeEntries: {
          include: {
            user: {
              select: { name: true, email: true },
            },
          },
          orderBy: { createdAt: "desc" },
        },
      },
    })

    return NextResponse.json(customers)
  } catch (error) {
    console.error("Error fetching customers:", error)
    return NextResponse.json({ error: "Failed to fetch customers" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { name, email, phone, company, address, notes } = body

    if (!name || (typeof name === 'string' && name.trim() === "")) {
      return NextResponse.json({ error: "Name is required" }, { status: 400 })
    }

    try {
      const customer = await prisma.customer.create({
        data: {
          name: typeof name === 'string' ? name.trim() : name,
          email: email && typeof email === 'string' && email.trim() !== "" ? email.trim() : null,
          phone: phone && typeof phone === 'string' && phone.trim() !== "" ? phone.trim() : null,
          company: company && typeof company === 'string' && company.trim() !== "" ? company.trim() : null,
          address: address && typeof address === 'string' && address.trim() !== "" ? address.trim() : null,
          notes: notes && typeof notes === 'string' && notes.trim() !== "" ? notes.trim() : null,
        },
      })

      return NextResponse.json(customer, { status: 201 })
    } catch (dbError: any) {
      console.error("Database error creating customer:", dbError)
      const errorMessage = dbError?.message || "Database error occurred"
      return NextResponse.json(
        { 
          error: errorMessage,
          code: dbError?.code,
          details: process.env.NODE_ENV === 'development' ? dbError?.stack : undefined
        }, 
        { status: 500 }
      )
    }
  } catch (error: any) {
    console.error("Error in POST /api/customers:", error)
    const errorMessage = error?.message || error?.toString() || "Failed to process request"
    return NextResponse.json(
      { 
        error: errorMessage,
        details: process.env.NODE_ENV === 'development' && error instanceof Error ? error.stack : undefined
      }, 
      { status: 500 }
    )
  }
}

