import { NextRequest, NextResponse } from "next/server"
import { prisma } from "@/lib/prisma"

export async function GET(request: NextRequest) {
  try {
    const searchParams = request.nextUrl.searchParams
    const customerId = searchParams.get("customerId")
    const userId = searchParams.get("userId")

    const timeEntries = await prisma.timeEntry.findMany({
      where: {
        ...(customerId ? { customerId } : {}),
        ...(userId ? { userId } : {}),
      },
      include: {
        customer: {
          select: { id: true, name: true, company: true },
        },
        user: {
          select: { id: true, name: true, email: true },
        },
      },
      orderBy: { createdAt: "desc" },
    })

    return NextResponse.json(timeEntries)
  } catch (error) {
    console.error("Error fetching time entries:", error)
    return NextResponse.json({ error: "Failed to fetch time entries" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { description, startTime, endTime, duration, customerId, userId } = body

    if (!description || !customerId || !userId) {
      return NextResponse.json(
        { error: "Description, customerId, and userId are required" },
        { status: 400 }
      )
    }

    const timeEntry = await prisma.timeEntry.create({
      data: {
        description,
        startTime: new Date(startTime),
        endTime: new Date(endTime),
        duration: parseInt(duration),
        customerId,
        userId,
      },
      include: {
        customer: {
          select: { id: true, name: true, company: true },
        },
        user: {
          select: { id: true, name: true, email: true },
        },
      },
    })

    return NextResponse.json(timeEntry, { status: 201 })
  } catch (error) {
    console.error("Error creating time entry:", error)
    return NextResponse.json({ error: "Failed to create time entry" }, { status: 500 })
  }
}

