import { NextRequest, NextResponse } from "next/server"
import { query } from "@/lib/db"
import { generateId, rowToTimeEntry } from "@/lib/utils/db-helpers"

export async function GET(request: NextRequest) {
  try {
    const searchParams = request.nextUrl.searchParams
    const customerId = searchParams.get("customerId")
    const userId = searchParams.get("userId")

    let timeEntriesQuery = `
      SELECT te.*,
        json_build_object(
          'id', c.id,
          'name', c.name,
          'company', c.company
        ) as customer,
        json_build_object(
          'id', u.id,
          'name', u.name,
          'email', u.email
        ) as user
      FROM time_entries te
      LEFT JOIN customers c ON c.id = te."customerId"
      LEFT JOIN users u ON u.id = te."userId"
      WHERE 1=1
    `

    const params: string[] = []
    let paramCount = 0

    if (customerId) {
      paramCount++
      timeEntriesQuery += ` AND te."customerId" = $${paramCount}`
      params.push(customerId)
    }

    if (userId) {
      paramCount++
      timeEntriesQuery += ` AND te."userId" = $${paramCount}`
      params.push(userId)
    }

    timeEntriesQuery += ` ORDER BY te."createdAt" DESC`

    const result = await query(
      timeEntriesQuery,
      params.length > 0 ? params : undefined
    )

    const timeEntries = result.rows.map((row: { customer: unknown; user: unknown }) => ({
      ...rowToTimeEntry(row),
      customer: typeof row.customer === 'string' ? JSON.parse(row.customer) : row.customer,
      user: typeof row.user === 'string' ? JSON.parse(row.user) : row.user,
    }))

    return NextResponse.json(timeEntries)
  } catch (error) {
    console.error("Error fetching time entries:", error)
    return NextResponse.json({ error: "Failed to fetch time entries" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { description, startTime, endTime, duration, customerId, userId } = body

    if (!description || !customerId || !userId) {
      return NextResponse.json(
        { error: "Description, customerId, and userId are required" },
        { status: 400 }
      )
    }

    const id = generateId()
    const now = new Date()

    const result = await query(
      `INSERT INTO time_entries (id, description, "startTime", "endTime", duration, "customerId", "userId", "createdAt", "updatedAt")
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
       RETURNING *`,
      [
        id,
        description,
        new Date(startTime),
        new Date(endTime),
        parseInt(String(duration), 10),
        customerId,
        userId,
        now,
        now,
      ]
    )

    const timeEntry = rowToTimeEntry(result.rows[0])

    // Get related customer and user
    const customerResult = await query(
      `SELECT id, name, company FROM customers WHERE id = $1`,
      [customerId]
    )
    const userResult = await query(
      `SELECT id, name, email FROM users WHERE id = $1`,
      [userId]
    )

    return NextResponse.json({
      ...timeEntry,
      customer: customerResult.rows[0] || null,
      user: userResult.rows[0] || null,
    }, { status: 201 })
  } catch (error) {
    console.error("Error creating time entry:", error)
    return NextResponse.json({ error: "Failed to create time entry" }, { status: 500 })
  }
}
