// WireGuard Easy API Client

// WG API URL Konfiguration
// Wenn Frontend und Backend in der gleichen CapRover App laufen:
// 
// Option 1: Backend läuft als separater CapRover Service mit eigener Domain
//   - Setze WG_API_URL auf die externe Backend-URL (z.B. https://wg-easy-backend.corexmanagement.de)
//
// Option 2: Backend läuft im gleichen Docker-Netzwerk (nicht empfohlen für CapRover)
//   - Verwende Docker Service-Namen oder interne IP
//
// Option 3: Backend läuft auf separatem Port (nur wenn beide im gleichen Container)
//   - localhost:51821
//
// WICHTIG: In CapRover sollte das Backend als eigener Service laufen mit eigener Domain!
// Setze WG_API_URL in CapRover als Environment Variable auf die Backend-URL
const WG_API_URL = process.env.WG_API_URL || 'https://corex-wg-easy.corexmanagement.de'
const WG_PASSWORD = process.env.WG_PASSWORD || 'CoreX2024!Secure#VPN'

interface WgClient {
  id: string
  name: string
  enabled: boolean
  address: string
  publicKey: string
  createdAt: string
  updatedAt: string
  latestHandshakeAt: string
  transferRx: number
  transferTx: number
  persistentKeepalive: string
}

interface WgServer {
  address: string
  listenPort: number
  publicKey: string
  endpoint: string
  dns: string
  allowedIPs: string
  createdAt: string
}

interface WgStats {
  clients: WgClient[]
  server: WgServer
}

interface CreateClientRequest {
  name: string
  allowedIPs?: string
}

interface CreateClientResponse {
  id: string
  name: string
  config: string
  qrCode: string
}

export class WgApiClient {
  private baseUrl: string
  private password: string

  constructor(baseUrl: string = WG_API_URL, password: string = WG_PASSWORD) {
    this.baseUrl = baseUrl.replace(/\/$/, '')
    this.password = password
  }

  private async request<T>(endpoint: string, options: RequestInit = {}): Promise<T> {
    const url = `${this.baseUrl}${endpoint}`
    
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      ...(options.headers as Record<string, string> || {}),
    }

    // Add basic auth if password is set
    if (this.password) {
      headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
    }

    console.log(`[WG API] Requesting: ${url}`)

    try {
      const response = await fetch(url, {
        ...options,
        headers,
      })

      if (!response.ok) {
        const errorText = await response.text()
        // Check if it's HTML (404 page) - means endpoint doesn't exist
        if (errorText.includes('<!DOCTYPE html>') || errorText.includes('<html')) {
          throw new Error(`WG API Endpoint not found: ${endpoint} (404). Check if WG_API_URL is correct: ${this.baseUrl}`)
        }
        throw new Error(`WG API Error: ${response.status} ${errorText.substring(0, 200)}`)
      }

      const contentType = response.headers.get('content-type')
      if (contentType && contentType.includes('application/json')) {
        return response.json()
      }
      
      // If not JSON, return as text
      return response.text() as unknown as T
    } catch (error) {
      // Handle network errors
      if (error instanceof TypeError && error.message.includes('fetch failed')) {
        throw new Error(`Failed to connect to WG API at ${this.baseUrl}. Please check if the backend is running and WG_API_URL is correct. Original error: ${error.message}`)
      }
      // Re-throw other errors
      throw error
    }
  }

  async getStats(): Promise<WgStats> {
    try {
      return await this.request<WgStats>('/api/sessions')
    } catch (error) {
      // If API is not available, return empty stats
      console.warn('WG API not available, returning empty stats:', error)
      return {
        clients: [],
        server: {
          address: '',
          listenPort: 0,
          publicKey: '',
          endpoint: '',
          dns: '',
          allowedIPs: '',
          createdAt: new Date().toISOString(),
        },
      }
    }
  }

  async getClients(): Promise<WgClient[]> {
    const stats = await this.getStats()
    return stats.clients || []
  }

  async getServer(): Promise<WgServer> {
    const stats = await this.getStats()
    return stats.server
  }

  async createClient(data: CreateClientRequest): Promise<CreateClientResponse> {
    // wg-easy uses /api/users endpoint
    return await this.request<CreateClientResponse>('/api/users', {
      method: 'POST',
      body: JSON.stringify(data),
    })
  }

  async deleteClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}`, {
      method: 'DELETE',
    })
  }

  async enableClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}/enable`, {
      method: 'PUT',
    })
  }

  async disableClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}/disable`, {
      method: 'PUT',
    })
  }

  async getClientConfig(id: string): Promise<string> {
    const headers: Record<string, string> = {}
    if (this.password) {
      headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
    }
    
    const url = `${this.baseUrl}/api/users/${id}/configuration`
    console.log(`[WG API] Fetching config from: ${url}`)
    
    const response = await fetch(url, {
      headers,
    })
    
    if (!response.ok) {
      const errorText = await response.text()
      if (errorText.includes('<!DOCTYPE html>') || errorText.includes('<html')) {
        throw new Error(`Config endpoint not found (404). Check if WG_API_URL is correct: ${this.baseUrl}`)
      }
      throw new Error(`Failed to get config: ${response.status} ${errorText.substring(0, 200)}`)
    }
    
    return response.text()
  }

  async getClientQR(id: string): Promise<string> {
    const headers: Record<string, string> = {}
    if (this.password) {
      headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
    }
    
    const url = `${this.baseUrl}/api/users/${id}/qr-code`
    console.log(`[WG API] Fetching QR code from: ${url}`)
    
    const response = await fetch(url, {
      headers,
    })
    
    if (!response.ok) {
      const errorText = await response.text()
      if (errorText.includes('<!DOCTYPE html>') || errorText.includes('<html')) {
        throw new Error(`QR code endpoint not found (404). Check if WG_API_URL is correct: ${this.baseUrl}`)
      }
      throw new Error(`Failed to get QR code: ${response.status} ${errorText.substring(0, 200)}`)
    }
    
    return response.text()
  }
}

export const wgApi = new WgApiClient()

