"use client"

import { useState, useEffect } from "react"
import { Logo } from "@/components/Logo"
import { ThemeToggle } from "@/components/theme-toggle"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Shield, Users, Network, Download, Plus, Trash2, Power, PowerOff, QrCode, FileDown } from "lucide-react"

interface Client {
  id: string
  name: string
  enabled: boolean
  address: string
  publicKey: string
  createdAt: string
  updatedAt: string
  latestHandshakeAt: string
  transferRx: number
  transferTx: number
  persistentKeepalive: string
}

interface Server {
  address: string
  listenPort: number
  publicKey: string
  endpoint: string
  dns: string
  allowedIPs: string
  createdAt: string
}

export default function Dashboard() {
  const [clients, setClients] = useState<Client[]>([])
  const [server, setServer] = useState<Server | null>(null)
  const [loading, setLoading] = useState(true)
  const [showAddClient, setShowAddClient] = useState(false)
  const [newClientName, setNewClientName] = useState("")
  const [isCreating, setIsCreating] = useState(false)

  useEffect(() => {
    fetchStats()
    // Refresh every 10 seconds
    const interval = setInterval(fetchStats, 10000)
    return () => clearInterval(interval)
  }, [])

  const fetchStats = async () => {
    try {
      const response = await fetch('/api/stats')
      if (response.ok) {
        const data = await response.json()
        setClients(data.clients || [])
        setServer(data.server || null)
      }
    } catch (error) {
      console.error("Error fetching stats:", error)
    } finally {
      setLoading(false)
    }
  }

  const handleCreateClient = async () => {
    if (!newClientName.trim()) return

    setIsCreating(true)
    try {
      const response = await fetch('/api/clients', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ name: newClientName.trim() }),
      })

      if (response.ok) {
        const client = await response.json()
        setNewClientName("")
        setShowAddClient(false)
        await fetchStats()
        // Show success message or download config
        alert(`Client "${client.name}" erstellt!`)
      } else {
        const error = await response.json()
        alert(`Fehler: ${error.error}`)
      }
    } catch (error) {
      console.error("Error creating client:", error)
      alert("Fehler beim Erstellen des Clients")
    } finally {
      setIsCreating(false)
    }
  }

  const handleDeleteClient = async (id: string, name: string) => {
    if (!confirm(`Möchten Sie den Client "${name}" wirklich löschen?`)) return

    try {
      const response = await fetch(`/api/clients/${id}`, {
        method: 'DELETE',
      })

      if (response.ok) {
        await fetchStats()
      } else {
        const error = await response.json()
        alert(`Fehler: ${error.error}`)
      }
    } catch (error) {
      console.error("Error deleting client:", error)
      alert("Fehler beim Löschen des Clients")
    }
  }

  const handleToggleClient = async (id: string, enabled: boolean) => {
    try {
      const response = await fetch(`/api/clients/${id}`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ enabled: !enabled }),
      })

      if (response.ok) {
        await fetchStats()
      } else {
        const error = await response.json()
        alert(`Fehler: ${error.error}`)
      }
    } catch (error) {
      console.error("Error toggling client:", error)
      alert("Fehler beim Ändern des Client-Status")
    }
  }

  const handleDownloadConfig = (id: string, name: string) => {
    window.open(`/api/clients/${id}/config`, '_blank')
  }

  const handleShowQR = (id: string) => {
    window.open(`/api/clients/${id}/qr`, '_blank')
  }

  const formatBytes = (bytes: number) => {
    if (bytes === 0) return '0 B'
    const k = 1024
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB']
    const i = Math.floor(Math.log(bytes) / Math.log(k))
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i]
  }

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Nie'
    const date = new Date(dateString)
    return date.toLocaleString('de-DE')
  }

  const activeClients = clients.filter(c => c.enabled).length
  const totalRx = clients.reduce((sum, c) => sum + c.transferRx, 0)
  const totalTx = clients.reduce((sum, c) => sum + c.transferTx, 0)
  const totalTraffic = totalRx + totalTx

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b bg-card">
        <div className="container mx-auto flex h-16 items-center justify-between px-4">
          <Logo className="h-8" />
          <div className="flex items-center gap-4">
            <span className="text-sm text-muted-foreground">WireGuard VPN Management</span>
            <ThemeToggle />
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="container mx-auto px-4 py-8">
        <div className="mb-8 flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold">WireGuard VPN Management</h1>
            <p className="text-muted-foreground mt-2">
              Verwalten Sie Ihre WireGuard VPN-Konfigurationen
            </p>
          </div>
          <Button onClick={() => setShowAddClient(!showAddClient)}>
            <Plus className="h-4 w-4 mr-2" />
            Neuer Client
          </Button>
        </div>

        {/* Add Client Form */}
        {showAddClient && (
          <Card className="mb-8">
            <CardHeader>
              <CardTitle>Neuer Client hinzufügen</CardTitle>
              <CardDescription>Erstellen Sie einen neuen WireGuard VPN Client</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label htmlFor="clientName">Client Name</Label>
                <Input
                  id="clientName"
                  placeholder="z.B. Laptop Max Mustermann"
                  value={newClientName}
                  onChange={(e) => setNewClientName(e.target.value)}
                  onKeyDown={(e) => {
                    if (e.key === 'Enter') handleCreateClient()
                  }}
                />
              </div>
              <div className="flex gap-2">
                <Button onClick={handleCreateClient} disabled={isCreating || !newClientName.trim()}>
                  {isCreating ? 'Wird erstellt...' : 'Client erstellen'}
                </Button>
                <Button variant="outline" onClick={() => {
                  setShowAddClient(false)
                  setNewClientName("")
                }}>
                  Abbrechen
                </Button>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Stats Cards */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4 mb-8">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Aktive Clients</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{activeClients} / {clients.length}</div>
              <p className="text-xs text-muted-foreground">Verbundene Clients</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Server Status</CardTitle>
              <Shield className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{server ? 'Online' : 'Offline'}</div>
              <p className="text-xs text-muted-foreground">{server ? `Port: ${server.listenPort}` : 'Server nicht erreichbar'}</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Gesamt Traffic</CardTitle>
              <Network className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{formatBytes(totalTraffic)}</div>
              <p className="text-xs text-muted-foreground">↑ {formatBytes(totalTx)} / ↓ {formatBytes(totalRx)}</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Clients gesamt</CardTitle>
              <Download className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{clients.length}</div>
              <p className="text-xs text-muted-foreground">Konfigurationen</p>
            </CardContent>
          </Card>
        </div>

        {/* Clients List */}
        <Card>
          <CardHeader>
            <CardTitle>VPN Clients</CardTitle>
            <CardDescription>Verwalten Sie Ihre WireGuard VPN Clients</CardDescription>
          </CardHeader>
          <CardContent>
            {loading ? (
              <div className="text-center py-8 text-muted-foreground">Lade Daten...</div>
            ) : clients.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                Noch keine Clients vorhanden. Erstellen Sie einen neuen Client.
              </div>
            ) : (
              <div className="space-y-4">
                {clients.map((client) => (
                  <div
                    key={client.id}
                    className="flex items-center justify-between p-4 border rounded-lg hover:bg-accent/50 transition-colors"
                  >
                    <div className="flex-1">
                      <div className="flex items-center gap-3">
                        <div className={`w-3 h-3 rounded-full ${client.enabled ? 'bg-green-500' : 'bg-gray-400'}`} />
                        <div>
                          <h3 className="font-semibold">{client.name}</h3>
                          <div className="text-sm text-muted-foreground space-y-1">
                            <div>Adresse: {client.address}</div>
                            <div>Letzte Verbindung: {formatDate(client.latestHandshakeAt)}</div>
                            <div>
                              Traffic: ↑ {formatBytes(client.transferTx)} / ↓ {formatBytes(client.transferRx)}
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleToggleClient(client.id, client.enabled)}
                        title={client.enabled ? 'Client deaktivieren' : 'Client aktivieren'}
                      >
                        {client.enabled ? (
                          <PowerOff className="h-4 w-4" />
                        ) : (
                          <Power className="h-4 w-4" />
                        )}
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleShowQR(client.id)}
                        title="QR-Code anzeigen"
                      >
                        <QrCode className="h-4 w-4" />
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleDownloadConfig(client.id, client.name)}
                        title="Konfiguration herunterladen"
                      >
                        <FileDown className="h-4 w-4" />
                      </Button>
                      <Button
                        variant="destructive"
                        size="sm"
                        onClick={() => handleDeleteClient(client.id, client.name)}
                        title="Client löschen"
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </main>
    </div>
  )
}
