// WireGuard Easy API Client

// WG API URL Konfiguration - Automatische Erkennung
// Versucht verschiedene URLs, falls keine Environment Variable gesetzt ist
function getWgApiUrl(): string {
  if (process.env.WG_API_URL) {
    return process.env.WG_API_URL
  }
  
  // Standard-URL (wg-easy Backend läuft normalerweise auf Port 51821)
  // In CapRover: Wenn beide in der gleichen App, versuche verschiedene Optionen
  return 'http://localhost:51821'
}

const WG_API_URL = getWgApiUrl()
const WG_PASSWORD = process.env.WG_PASSWORD || 'CoreX2024!Secure#VPN'

interface WgClient {
  id: string
  name: string
  enabled: boolean
  address: string
  publicKey: string
  createdAt: string
  updatedAt: string
  latestHandshakeAt: string
  transferRx: number
  transferTx: number
  persistentKeepalive: string
}

interface WgServer {
  address: string
  listenPort: number
  publicKey: string
  endpoint: string
  dns: string
  allowedIPs: string
  createdAt: string
}

interface WgStats {
  clients: WgClient[]
  server: WgServer
}

interface CreateClientRequest {
  name: string
  allowedIPs?: string
}

interface CreateClientResponse {
  id: string
  name: string
  config: string
  qrCode: string
}

export class WgApiClient {
  private baseUrl: string
  private password: string

  constructor(baseUrl: string = WG_API_URL, password: string = WG_PASSWORD) {
    this.baseUrl = baseUrl.replace(/\/$/, '')
    this.password = password
  }

  private async request<T>(endpoint: string, options: RequestInit = {}): Promise<T> {
    // Liste möglicher Backend-URLs zum Ausprobieren
    // Hinweis: Port 51821 ist der Standard-Port für wg-easy Web-UI
    // Wenn das Backend auf einem anderen Port läuft, muss WG_API_URL als Environment Variable gesetzt werden
    const possibleUrls = [
      this.baseUrl, // Zuerst die konfigurierte URL versuchen
      'http://localhost:51821',
      'http://127.0.0.1:51821',
      'http://localhost:51820', // Alternative Port
      'http://wg-easy:51821',
      'http://wg-easy:51820',
      'http://srv-captain--wg-easy:51821',
      'http://srv-captain--wg-easy:51820',
      // Versuche auch mögliche CapRover Service-Namen
      'http://srv-captain--corex-wg-easy:51821',
      'http://srv-captain--corex-wg-easy:51820',
    ]

    let lastError: Error | null = null

    for (const baseUrl of possibleUrls) {
      const url = `${baseUrl}${endpoint}`
      
      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
        ...(options.headers as Record<string, string> || {}),
      }

      // Add basic auth if password is set
      if (this.password) {
        headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
      }

      console.log(`[WG API] Trying: ${url}`)

      try {
        const response = await fetch(url, {
          ...options,
          headers,
        })

        if (!response.ok) {
          const errorText = await response.text()
          // Check if it's HTML (404 page) - means endpoint doesn't exist
          if (errorText.includes('<!DOCTYPE html>') || errorText.includes('<html')) {
            // Try next URL
            continue
          }
          // Other HTTP errors - try next URL
          continue
        }

        // Success! Update baseUrl for future requests
        if (baseUrl !== this.baseUrl) {
          console.log(`[WG API] Successfully connected to: ${baseUrl}`)
          this.baseUrl = baseUrl
        }

        const contentType = response.headers.get('content-type')
        if (contentType && contentType.includes('application/json')) {
          return response.json()
        }
        
        // If not JSON, return as text
        return response.text() as unknown as T
      } catch (error) {
        // Network error - try next URL
        lastError = error instanceof Error ? error : new Error(String(error))
        continue
      }
    }

    // All URLs failed - provide helpful error message
    const triedUrls = possibleUrls.filter((url, index, self) => self.indexOf(url) === index) // Remove duplicates
    throw new Error(
      `Failed to connect to WG API Backend. ` +
      `Tried URLs: ${triedUrls.join(', ')}. ` +
      `Last error: ${lastError?.message || 'Unknown error'}. ` +
      `Please ensure the wg-easy backend is running and set WG_API_URL environment variable if it's on a different URL/port.`
    )
  }

  async getStats(): Promise<WgStats> {
    try {
      return await this.request<WgStats>('/api/sessions')
    } catch (error) {
      // If API is not available, return empty stats
      console.warn('WG API not available, returning empty stats:', error)
      return {
        clients: [],
        server: {
          address: '',
          listenPort: 0,
          publicKey: '',
          endpoint: '',
          dns: '',
          allowedIPs: '',
          createdAt: new Date().toISOString(),
        },
      }
    }
  }

  async getClients(): Promise<WgClient[]> {
    const stats = await this.getStats()
    return stats.clients || []
  }

  async getServer(): Promise<WgServer> {
    const stats = await this.getStats()
    return stats.server
  }

  async createClient(data: CreateClientRequest): Promise<CreateClientResponse> {
    // wg-easy uses /api/users endpoint
    return await this.request<CreateClientResponse>('/api/users', {
      method: 'POST',
      body: JSON.stringify(data),
    })
  }

  async deleteClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}`, {
      method: 'DELETE',
    })
  }

  async enableClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}/enable`, {
      method: 'PUT',
    })
  }

  async disableClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}/disable`, {
      method: 'PUT',
    })
  }

  async getClientConfig(id: string): Promise<string> {
    const headers: Record<string, string> = {}
    if (this.password) {
      headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
    }
    
    const url = `${this.baseUrl}/api/users/${id}/configuration`
    console.log(`[WG API] Fetching config from: ${url}`)
    
    const response = await fetch(url, {
      headers,
    })
    
    if (!response.ok) {
      const errorText = await response.text()
      if (errorText.includes('<!DOCTYPE html>') || errorText.includes('<html')) {
        throw new Error(`Config endpoint not found (404). Check if WG_API_URL is correct: ${this.baseUrl}`)
      }
      throw new Error(`Failed to get config: ${response.status} ${errorText.substring(0, 200)}`)
    }
    
    return response.text()
  }

  async getClientQR(id: string): Promise<string> {
    const headers: Record<string, string> = {}
    if (this.password) {
      headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
    }
    
    const url = `${this.baseUrl}/api/users/${id}/qr-code`
    console.log(`[WG API] Fetching QR code from: ${url}`)
    
    const response = await fetch(url, {
      headers,
    })
    
    if (!response.ok) {
      const errorText = await response.text()
      if (errorText.includes('<!DOCTYPE html>') || errorText.includes('<html')) {
        throw new Error(`QR code endpoint not found (404). Check if WG_API_URL is correct: ${this.baseUrl}`)
      }
      throw new Error(`Failed to get QR code: ${response.status} ${errorText.substring(0, 200)}`)
    }
    
    return response.text()
  }
}

export const wgApi = new WgApiClient()

