// WireGuard Easy API Client

const WG_API_URL = process.env.WG_API_URL || 'http://localhost:51821'
const WG_PASSWORD = process.env.WG_PASSWORD || ''

interface WgClient {
  id: string
  name: string
  enabled: boolean
  address: string
  publicKey: string
  createdAt: string
  updatedAt: string
  latestHandshakeAt: string
  transferRx: number
  transferTx: number
  persistentKeepalive: string
}

interface WgServer {
  address: string
  listenPort: number
  publicKey: string
  endpoint: string
  dns: string
  allowedIPs: string
  createdAt: string
}

interface WgStats {
  clients: WgClient[]
  server: WgServer
}

interface CreateClientRequest {
  name: string
  allowedIPs?: string
}

interface CreateClientResponse {
  id: string
  name: string
  config: string
  qrCode: string
}

export class WgApiClient {
  private baseUrl: string
  private password: string

  constructor(baseUrl: string = WG_API_URL, password: string = WG_PASSWORD) {
    this.baseUrl = baseUrl.replace(/\/$/, '')
    this.password = password
  }

  private async request<T>(endpoint: string, options: RequestInit = {}): Promise<T> {
    const url = `${this.baseUrl}${endpoint}`
    
    const headers = {
      'Content-Type': 'application/json',
      ...options.headers,
    }

    // Add basic auth if password is set
    if (this.password) {
      headers['Authorization'] = `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
    }

    const response = await fetch(url, {
      ...options,
      headers,
    })

    if (!response.ok) {
      const errorText = await response.text()
      throw new Error(`WG API Error: ${response.status} ${errorText}`)
    }

    return response.json()
  }

  async getStats(): Promise<WgStats> {
    return this.request<WgStats>('/api/sessions')
  }

  async getClients(): Promise<WgClient[]> {
    const stats = await this.getStats()
    return stats.clients || []
  }

  async getServer(): Promise<WgServer> {
    const stats = await this.getStats()
    return stats.server
  }

  async createClient(data: CreateClientRequest): Promise<CreateClientResponse> {
    return this.request<CreateClientResponse>('/api/users', {
      method: 'POST',
      body: JSON.stringify(data),
    })
  }

  async deleteClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}`, {
      method: 'DELETE',
    })
  }

  async enableClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}/enable`, {
      method: 'PUT',
    })
  }

  async disableClient(id: string): Promise<void> {
    await this.request(`/api/users/${id}/disable`, {
      method: 'PUT',
    })
  }

  async getClientConfig(id: string): Promise<string> {
    const response = await fetch(`${this.baseUrl}/api/users/${id}/configuration`, {
      headers: this.password ? {
        'Authorization': `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
      } : {},
    })
    
    if (!response.ok) {
      throw new Error(`Failed to get config: ${response.status}`)
    }
    
    return response.text()
  }

  async getClientQR(id: string): Promise<string> {
    const response = await fetch(`${this.baseUrl}/api/users/${id}/qr-code`, {
      headers: this.password ? {
        'Authorization': `Basic ${Buffer.from(`:${this.password}`).toString('base64')}`
      } : {},
    })
    
    if (!response.ok) {
      throw new Error(`Failed to get QR code: ${response.status}`)
    }
    
    return response.text()
  }
}

export const wgApi = new WgApiClient()

