import { NextRequest, NextResponse } from "next/server"
import { wgApi } from "@/lib/wg-api"

export async function GET() {
  try {
    const clients = await wgApi.getClients()
    return NextResponse.json(clients)
  } catch (error) {
    console.error("Error fetching clients:", error)
    const errorMessage = error instanceof Error ? error.message : "Failed to fetch clients"
    return NextResponse.json(
      { error: errorMessage },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { name, allowedIPs } = body

    if (!name || typeof name !== 'string' || name.trim() === '') {
      return NextResponse.json(
        { error: "Client name is required" },
        { status: 400 }
      )
    }

    try {
      const client = await wgApi.createClient({
        name: name.trim(),
        allowedIPs: allowedIPs || undefined,
      })

      return NextResponse.json(client, { status: 201 })
    } catch (apiError) {
      console.error("WG API Error:", apiError)
      const errorMessage = apiError instanceof Error ? apiError.message : "Failed to create client"
      
      // Check if it's a 404 - API endpoint might be wrong
      if (errorMessage.includes('404') || errorMessage.includes('not found')) {
        return NextResponse.json(
          { 
            error: "WireGuard API endpoint not found. Please check WG_API_URL configuration.",
            details: errorMessage
          },
          { status: 503 }
        )
      }
      
      return NextResponse.json(
        { error: errorMessage },
        { status: 500 }
      )
    }
  } catch (error) {
    console.error("Error creating client:", error)
    const errorMessage = error instanceof Error ? error.message : "Failed to create client"
    return NextResponse.json(
      { error: errorMessage },
      { status: 500 }
    )
  }
}

