import { NextRequest, NextResponse } from "next/server"
import { wgApi } from "@/lib/wg-api"

export async function GET() {
  try {
    const clients = await wgApi.getClients()
    return NextResponse.json(clients)
  } catch (error) {
    console.error("Error fetching clients:", error)
    const errorMessage = error instanceof Error ? error.message : "Failed to fetch clients"
    return NextResponse.json(
      { error: errorMessage },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { name, allowedIPs } = body

    if (!name || typeof name !== 'string' || name.trim() === '') {
      return NextResponse.json(
        { error: "Client name is required" },
        { status: 400 }
      )
    }

    const client = await wgApi.createClient({
      name: name.trim(),
      allowedIPs: allowedIPs || undefined,
    })

    return NextResponse.json(client, { status: 201 })
  } catch (error) {
    console.error("Error creating client:", error)
    const errorMessage = error instanceof Error ? error.message : "Failed to create client"
    return NextResponse.json(
      { error: errorMessage },
      { status: 500 }
    )
  }
}

